---
title: Schema Configurations
---

# Schema Configurations

Strawberry allows to customise how the schema is generated by passing
configurations.

To customise the schema you can create an instance of `StrawberryConfig`, as
shown in the example below:

```python
import strawberry

from strawberry.schema.config import StrawberryConfig


@strawberry.type
class Query:
    example_field: str


schema = strawberry.Schema(query=Query, config=StrawberryConfig(auto_camel_case=False))
```

In this case we are disabling the auto camel casing feature, so your output
schema will look like this:

```graphql
type Query {
  example_field: String!
}
```

## Available configurations

Here's a list of the available configurations:

### auto_camel_case

By default Strawberry will convert the field names to camel case, so a field
like `example_field` will be converted to `exampleField`. You can disable this
feature by setting `auto_camel_case` to `False`.

```python
schema = strawberry.Schema(query=Query, config=StrawberryConfig(auto_camel_case=False))
```

### default_resolver

By default Strawberry will use the `getattr` function as the default resolver.
You can customise this by setting the `default_resolver` configuration.

This can be useful in cases you want to allow returning a dictionary from a
resolver.

```python
import strawberry

from strawberry.schema.config import StrawberryConfig


def custom_resolver(obj, field):
    try:
        return obj[field]
    except (KeyError, TypeError):
        return getattr(obj, field)


@strawberry.type
class User:
    name: str


@strawberry.type
class Query:
    @strawberry.field
    def user(self, info) -> User:  # this won't type check, but will work at runtime
        return {"name": "Patrick"}


schema = strawberry.Schema(
    query=Query, config=StrawberryConfig(default_resolver=custom_resolver)
)
```

### relay_max_results

By default Strawberry's max limit for relay connections is 100. You can
customise this by setting the `relay_max_results` configuration.

```python
schema = strawberry.Schema(query=Query, config=StrawberryConfig(relay_max_results=50))
```

### disable_field_suggestions

By default Strawberry will suggest fields when a field is not found in the
schema. You can disable this feature by setting `disable_field_suggestions` to
`True`.

```python
schema = strawberry.Schema(
    query=Query, config=StrawberryConfig(disable_field_suggestions=True)
)
```

### info_class

By default Strawberry will create an object of type `strawberry.Info` when the
user defines `info: Info` as a parameter to a type or query. You can change this
behaviour by setting `info_class` to a subclass of `strawberry.Info`.

This can be useful when you want to create a simpler interface for info- or
context-based properties, or if you wanted to attach additional properties to
the `Info` class.

```python
class CustomInfo(Info):
    @property
    def response_headers(self) -> Headers:
        return self.context["response"].headers


schema = strawberry.Schema(query=Query, config=StrawberryConfig(info_class=CustomInfo))
```
