use core:lang;
use core:io;
use lang:bs;
use lang:bs:macro;
use core:asm;

// All functions here shall be executed on the compiler thread.
on Compiler:

/**
 * A decorator for types that adds members required for automatic serialization and deserialization.
 */
void serializable(Type type) {
	Value t = thisPtr(type);
	Value oStream(named{ObjOStream});
	Value iStream(named{ObjIStream});

	// Original functions.
	Function? oTypeFn = findFn(type, "serializedType", [], Value(named{SerializedType}));
	Function? oWriteFn = findFn(type, "write", [t, oStream], Value());
	Function? oReadCtor = findFn(type, "__init", [t, iStream], Value());
	Function? oReadFn = findFn(type, "read", [iStream], Value(type));

	Member[] members;
	if (oTypeFn.any & oWriteFn.any & oReadCtor.any) {
		// Ok.
	} else if (oWriteFn.any & oReadCtor.any) {
		// Ok, we can generate 'serializedType' in this case!
	} else if (oTypeFn.any | oWriteFn.any | oReadCtor.any) {
		// Bad, we can't generate just one of 'write' and 'read', especially if the type function is defined.
		StrBuf msg;
		msg << "You need to provide either both a write function and a read constructor or none of them for ";
		msg << type.identifier << ". The serializedType function is recommended, but not necessary. Missing:";
		if (oTypeFn.empty)
			msg << " serializedType";
		if (oWriteFn.empty)
			msg << " write";
		if (oReadCtor.empty)
			msg << " read constructor";
		msg << ".";

		throw SyntaxError(type.pos, msg.toS);
	} else {
		// Ok! None of the members are present!
		members = serializedMembers(type);
	}

	Function readCtor = if (oReadCtor) {
		oReadCtor;
	} else {
		addCtor(type, members);
	};

	Function typeFn = if (oTypeFn) {
		oTypeFn;
	} else {
		// Add the proper type function.
		if (oReadCtor.any | oWriteFn.any)
			addCustomType(type, readCtor);
		else
			addStdType(type, members, readCtor);
	};

	Function writeFn = if (oWriteFn) {
		oWriteFn;
	} else {
		addWrite(type, members, typeFn);
	};

	if (!oReadFn.any) {
		// Generate the read function, just to be nice!
		addRead(type);
	}
}

private:

// Find a function inside 'type', and make sure it has the proper return type.
Function? findFn(Type type, Str name, Value[] params, Value result) {
	if (fn = type.findHere(SimplePart(name, params), Scope(type)) as Function) {
		if (result.mayReferTo(fn.result)) {
			return fn;
		}
	}
	null;
}


/**
 * Information about a member to serialize.
 */
class Member {
	// Serialized name (could differ from the member's actual name).
	Str name;

	// The member to serialize.
	MemberVar member;

	// Create.
	init(Str name, MemberVar member) {
		init { name = name; member = member; }
	}

	// Find the initializer for this member.
	Function? initializer() {
		if (existing = member.initializer) {
			return existing;
		} else {
			return createDefaultInitializer(member);
		}
	}

	// To string.
	void toS(StrBuf to) {
		to << name;
	}
}

// Find all members to serialize.
Member[] serializedMembers(Type t) {
	// TODO: It should be possible to mark variables for inclusion or exclusion by including other annotations.

	Member[] result;
	for (m in t.variables)
		result << Member(m.name, m);

	// Note: Asking the type for variables is stable. Otherwise, we could sort by offset:
	// result.sort((a, b) => a.member.rawOffset.current() < b.member.rawOffset.current());

	result;
}

// Throw on failure.
LocalVar safeVariable(Block block, Str name) {
	if (x = block.variable(SimplePart(name))) {
		x;
	} else {
		throw SyntaxError(SrcPos(), "Internal error: the variable was not found!");
		// Should be removed when we have exceptions!
		LocalVar(name, Value(), SrcPos(), false);
	}
}

// Find a suitable read or write function for a type, as accessed from 'from'.
Function findReadFunction(SrcPos pos, Value type, Type from) {
	if (t = type.type) {
		Value param(named{ObjIStream});
		if (x = t.find(SimplePart("read", [param]), Scope(from)) as Function) {
			return x;
		}
	}

	throw SyntaxError(pos, "The type ${type} is not serializable.");
}


/**
 * Add the constructor for deserialization. Returns the created function.
 *
 * Roughly equivalent to the following:
 *
 * init(ObjIStream from) {
 *   init(from) {
 *     foo = T:read(from);
 *     bar = T:read(from);
 *     // ...
 *   }
 *   from.end();
 * }
 */
Function addCtor(Type type, Member[] members) {
	Value objStream(named{ObjIStream});
	SrcPos pos;

	BSTreeCtor ctor([thisParam(type), ValParam(objStream, "from")], pos);
	CtorBody body(ctor, Scope(type, BSLookup()));
	LocalVarAccess from(pos, body.safeVariable("from"));

	// TODO: We should make these into temporaries to ensure proper ordering?
	Initializer[] init;
	for (m in members) {
		Function toCall = findReadFunction(pos, m.member.type, type);
		FnCall expr(pos, body.scope, toCall, Actuals(from));
		init << Initializer(SStr(m.name), expr);
	}

	// We don't want to pass parameters to the superclass if it is Object or TObject.
	if (type.declaredSuper) {
		body.add(InitBlock(pos, body, Actuals(from), init));
	} else {
		body.add(InitBlock(pos, body, Actuals(), init));
	}

	// Call 'end'.
	body.add(namedExpr(body, pos, "end", from));

	ctor.body = body;

	// Should be protected; external actors should call "read" instead.
	ctor.visibility = typeProtected();

	type.add(ctor);
	ctor;
}

// Find the type description in the parent class.
SerializedType findSerializedType(Type type, Type from) {
	if (fn = type.findHere(SimplePart("serializedType"), Scope(from)) as Function) {
		if (ptr = fn.pointer as fn()->SerializedType) {
			return ptr.call();
		}
	}

	throw SyntaxError(SrcPos(), "The type ${type} is not serializable.");
}

/**
 * Add the member 'serializedType' for standard types.
 */
Function addStdType(Type type, Member[] members, Function ctorFun) {
	SerializedStdType desc = if (t = type.declaredSuper) {
		SerializedStdType(type, ctorFun.pointer, t);
	} else {
		SerializedStdType(type, ctorFun.pointer);
	};
	for (m in members) {
		if (t = m.member.type.type) {
			if (init = m.initializer) {
				desc.add(m.name, t, init.pointer);
			} else {
				desc.add(m.name, t);
			}
		}
	}

	addType(type, desc);
}

/**
 * Add the member 'serializedType' for custom types.
 */
Function addCustomType(Type type, Function ctorFun) {
	SerializedType desc = if (t = type.declaredSuper) {
		SerializedType(type, ctorFun.pointer, type);
	} else {
		SerializedType(type, ctorFun.pointer);
	};
	addType(type, desc);
}


/**
 * Add the member 'serializedType'. Returns the created function.
 */
Function addType(Type type, SerializedType desc) {
	Value result(named{SerializedType});

	Listing l(false, result.desc);
	l << prolog();
	l << fnRet(objPtr(desc));

	Function fun(result, "serializedType", []);
	fun.make(FnFlags:static);
	fun.setCode(DynamicCode(l));
	type.add(fun);
	fun;
}

/**
 * Add the member 'write'.
 *
 * Roughly equivalent to the following:
 *
 * void write(ObjOStream to) {
 *   if (to.startClass(desc, this)) {
 *     foo.write(to);
 *     bar.write(to);
 *     ...
 *     to.end();
 *   }
 * }
 */
Function addWrite(Type type, Member[] members, Function typeFun) {
	SrcPos pos;
	Value t = thisPtr(type);
	Value objStream(named{ObjOStream});
	BSTreeFn fun(Value(), SStr("write", pos), [thisParam(type), ValParam(objStream, "to")], null);
	FnBody body(fun, Scope(type, BSLookup()));
	LocalVarAccess me(pos, body.safeVariable("this"));
	LocalVarAccess to(pos, body.safeVariable("to"));

	// FnCall desc(pos, body.scope, typeFun, Actuals(), false, true);
	Expr startCall = if (!t.isObject()) {
		namedExpr(body, pos, "startValue", to, Actuals(ReferNamed(pos, type)));
	} else {
		Actuals params;
		params.add(ReferNamed(pos, type));
		params.add(me);
		namedExpr(body, pos, "startClass", to, params);
	};

	If check(body, startCall);
	ExprBlock writeCalls(pos, check.successBlock);
	check.success(writeCalls);
	body.add(check);

	if (s = type.declaredSuper) {
		// Call the super function.
		if (fn = s.findHere(SimplePart("write", [t, objStream]), body.scope) as Function) {
			Actuals a;
			a.add(me);
			a.add(to);
			writeCalls.add(FnCall(pos, body.scope, fn, a, false));
		} else {
			throw SyntaxError(pos, "The type ${s} does not have a write function!");
		}
	}

	for (m in members) {
		MemberVarAccess v(pos, me, m.member, true);
		writeCalls.add(namedExpr(writeCalls, pos, "write", v, Actuals(to)));
	}

	writeCalls.add(namedExpr(writeCalls, pos, "end", to));

	fun.body = body;
	type.add(fun);
	fun;
}


/**
 * Function implementing the 'read' member required for deserialization.
 *
 * Roughly equivalent to the following (explicitly circumventing some type-checks, as we know they
 * will never fail):
 *
 * T read(ObjIStream from) : static {
 *   from.readObject(named{T});
 * }
 */
void addRead(Type type) {
	Value t(type);
	Value objStream(named{ObjIStream});

	Listing l(false, t.desc);
	var stream = l.createParam(objStream.desc);
	l << prolog();

	if (t.isObject) {
		// Class or actor.
		var toCall = named{ObjIStream:readClass<ObjIStream, Type>}.ref;

		l << fnParam(objStream.desc, stream);
		l << fnParam(ptrDesc, type.typeRef);
		l << fnCall(toCall, true, t.desc, ptrA);
		l << fnRet(ptrA);
	} else {
		var tmp = l.createVar(l.root, t.size);
		var toCall = named{ObjIStream:readValue<ObjIStream, Type, core:lang:unknown:PTR_GC>}.ref;

		l << lea(ptrA, tmp);
		l << fnParam(objStream.desc, stream);
		l << fnParam(ptrDesc, type.typeRef);
		l << fnParam(ptrDesc, ptrA);
		l << fnCall(toCall, true);
		l << fnRet(tmp);
	}

	Function fun(t, "read", [objStream]);
	fun.make(FnFlags:static);
	fun.setCode(DynamicCode(l));
	type.add(fun);
}

/**
 * Create an initializer for a default-initializable type:
 */
Function? createDefaultInitializer(MemberVar var) {
	unless (type = var.type.type)
		return null; // Type is void.

	Scope scope(var);
	unless (ctor = type.find(SimplePart("__init", [type]), scope) as Function)
		return null; // No default ctor.

	// We can just return the constructor, it is compatible.
	return ctor;
}
